<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasOne;

class Booking extends Model
{
    protected $fillable = [
        'ride_id',
        'passenger_id',
        'booking_type',
        'number_of_seats',
        'total_price',
        'payment_status',
        'booking_status',
        'pickup_location',
        'drop_location',
        'passenger_notes',
    ];

    protected function casts(): array
    {
        return [
            'number_of_seats' => 'integer',
            'total_price' => 'decimal:2',
        ];
    }

    // Relationships

    /**
     * Get the ride for this booking
     */
    public function ride(): BelongsTo
    {
        return $this->belongsTo(Ride::class);
    }

    /**
     * Get the passenger who made this booking
     */
    public function passenger(): BelongsTo
    {
        return $this->belongsTo(User::class, 'passenger_id');
    }

    /**
     * Get the payment for this booking
     */
    public function payment(): HasOne
    {
        return $this->hasOne(Payment::class);
    }

    /**
     * Get the review for this booking
     */
    public function review(): HasOne
    {
        return $this->hasOne(Review::class);
    }

    // Scopes

    /**
     * Scope for confirmed bookings
     */
    public function scopeConfirmed($query)
    {
        return $query->where('booking_status', 'confirmed');
    }

    /**
     * Scope for pending bookings
     */
    public function scopePending($query)
    {
        return $query->where('booking_status', 'pending');
    }

    /**
     * Scope for paid bookings
     */
    public function scopePaid($query)
    {
        return $query->where('payment_status', 'paid');
    }

    // Helper Methods

    /**
     * Check if booking is confirmed
     */
    public function isConfirmed(): bool
    {
        return $this->booking_status === 'confirmed';
    }

    /**
     * Check if payment is completed
     */
    public function isPaid(): bool
    {
        return $this->payment_status === 'paid';
    }

    /**
     * Check if booking can be cancelled
     */
    public function isCancellable(): bool
    {
        return in_array($this->booking_status, ['pending', 'confirmed'])
            && $this->ride->departure_datetime > now()->addHours(2);
    }

    /**
     * Get booking type badge
     */
    public function getBookingTypeBadgeAttribute(): string
    {
        return $this->booking_type === 'full_car' 
            ? '🚗 Full Car' 
            : '👤 Seat Booking';
    }

    /**
     * Get status badge color
     */
    public function getStatusColorAttribute(): string
    {
        return match($this->booking_status) {
            'confirmed' => 'green',
            'pending' => 'yellow',
            'cancelled' => 'red',
            'completed' => 'blue',
            default => 'gray'
        };
    }
}
