<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Database\Eloquent\Relations\HasMany;

class User extends Authenticatable
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasFactory, Notifiable;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'phone',
        'avatar',
        'user_type',
        'is_verified',
        'is_admin',
        'verification_document',
        'rating',
        'total_rides',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
            'is_verified' => 'boolean',
            'rating' => 'decimal:2',
            'total_rides' => 'integer',
        ];
    }

    // Relationships

    /**
     * Get the vehicles owned by the user (for drivers)
     */
    public function vehicles(): HasMany
    {
        return $this->hasMany(Vehicle::class, 'driver_id');
    }

    /**
     * Get the rides created by this user (as driver)
     */
    public function ridesAsDriver(): HasMany
    {
        return $this->hasMany(Ride::class, 'driver_id');
    }

    /**
     * Get the bookings made by this user (as passenger)
     */
    public function bookingsAsPassenger(): HasMany
    {
        return $this->hasMany(Booking::class, 'passenger_id');
    }

    /**
     * Get reviews written by this user
     */
    public function reviewsGiven(): HasMany
    {
        return $this->hasMany(Review::class, 'reviewer_id');
    }

    /**
     * Get reviews received by this user
     */
    public function reviewsReceived(): HasMany
    {
        return $this->hasMany(Review::class, 'reviewee_id');
    }

    /**
     * Get notifications for this user
     */
    public function notifications(): HasMany
    {
        return $this->hasMany(Notification::class);
    }

    // Helper Methods

    /**
     * Check if user is a driver
     */
    public function isDriver(): bool
    {
        return in_array($this->user_type, ['driver', 'both']);
    }

    /**
     * Check if user is a passenger
     */
    public function isPassenger(): bool
    {
        return in_array($this->user_type, ['passenger', 'both']);
    }

    /**
     * Get the user's display name with verification status
     */
    public function getDisplayNameAttribute(): string
    {
        return $this->name . ($this->is_verified ? ' ✓' : '');
    }

    /**
     * Get the user's avatar URL
     */
    public function getAvatarUrlAttribute(): string
    {
        return $this->avatar 
            ? asset('storage/' . $this->avatar) 
            : 'https://ui-avatars.com/api/?name=' . urlencode($this->name) . '&background=4F46E5&color=fff';
    }
}
