<?php

namespace Database\Seeders;

use App\Models\{User, City, Vehicle, Route, Ride, Booking, Payment, Review, Notification};
use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Hash;

class DatabaseSeeder extends Seeder
{
    use WithoutModelEvents;

    /**
     * Seed the application's database.
     */
    public function run(): void
    {
        // Create Users
        $admin = User::create([
            'name' => 'Admin User',
            'email' => 'admin@ridepool.com',
            'password' => Hash::make('password'),
            'phone' => '+1234567890',
            'user_type' => 'both',
            'is_verified' => true,
            'is_admin' => true,
            'rating' => 4.8,
            'total_rides' => 0,
        ]);

        $driver1 = User::create([
            'name' => 'John Driver',
            'email' => 'john@driver.com',
            'password' => Hash::make('password'),
            'phone' => '+1234567891',
            'user_type' => 'driver',
            'is_verified' => true,
            'rating' => 4.9,
            'total_rides' => 45,
        ]);

        $driver2 = User::create([
            'name' => 'Sarah Williams',
            'email' => 'sarah@driver.com',
            'password' => Hash::make('password'),
            'phone' => '+1234567892',
            'user_type' => 'driver',
            'is_verified' => true,
            'rating' => 4.7,
            'total_rides' => 32,
        ]);

        $passenger1 = User::create([
            'name' => 'Mike Passenger',
            'email' => 'mike@passenger.com',
            'password' => Hash::make('password'),
            'phone' => '+1234567893',
            'user_type' => 'passenger',
            'is_verified' => true,
            'rating' => 4.6,
            'total_rides' => 12,
        ]);

        $passenger2 = User::create([
            'name' => 'Emma Johnson',
            'email' => 'emma@passenger.com',
            'password' => Hash::make('password'),
            'phone' => '+1234567894',
            'user_type' => 'passenger',
            'is_verified' => true,
            'rating' => 4.8,
            'total_rides' => 8,
        ]);

        // Create Cities - Pakistani Cities
        $cities = [
            ['name' => 'Karachi', 'state' => 'Sindh', 'country' => 'Pakistan', 'latitude' => 24.8607, 'longitude' => 67.0011, 'is_active' => true],
            ['name' => 'Lahore', 'state' => 'Punjab', 'country' => 'Pakistan', 'latitude' => 31.5204, 'longitude' => 74.3587, 'is_active' => true],
            ['name' => 'Islamabad', 'state' => 'Islamabad Capital Territory', 'country' => 'Pakistan', 'latitude' => 33.6844, 'longitude' => 73.0479, 'is_active' => true],
            ['name' => 'Rawalpindi', 'state' => 'Punjab', 'country' => 'Pakistan', 'latitude' => 33.5651, 'longitude' => 73.0169, 'is_active' => true],
            ['name' => 'Faisalabad', 'state' => 'Punjab', 'country' => 'Pakistan', 'latitude' => 31.4504, 'longitude' => 73.1350, 'is_active' => true],
            ['name' => 'Multan', 'state' => 'Punjab', 'country' => 'Pakistan', 'latitude' => 30.1575, 'longitude' => 71.5249, 'is_active' => true],
            ['name' => 'Peshawar', 'state' => 'Khyber Pakhtunkhwa', 'country' => 'Pakistan', 'latitude' => 34.0151, 'longitude' => 71.5249, 'is_active' => true],
            ['name' => 'Quetta', 'state' => 'Balochistan', 'country' => 'Pakistan', 'latitude' => 30.1798, 'longitude' => 66.9750, 'is_active' => true],
            ['name' => 'Sialkot', 'state' => 'Punjab', 'country' => 'Pakistan', 'latitude' => 32.4945, 'longitude' => 74.5229, 'is_active' => true],
            ['name' => 'Gujranwala', 'state' => 'Punjab', 'country' => 'Pakistan', 'latitude' => 32.1617, 'longitude' => 74.1883, 'is_active' => true],
        ];

        foreach ($cities as $city) {
            City::create($city);
        }

        // Get cities for routes
        $karachi = City::where('name', 'Karachi')->first();
        $lahore = City::where('name', 'Lahore')->first();
        $islamabad = City::where('name', 'Islamabad')->first();
        $rawalpindi = City::where('name', 'Rawalpindi')->first();
        $faisalabad = City::where('name', 'Faisalabad')->first();
        $multan = City::where('name', 'Multan')->first();

        // Create Vehicles with images
        $vehicle1 = Vehicle::create([
            'driver_id' => $driver1->id,
            'vehicle_type' => 'sedan',
            'brand' => 'Toyota',
            'model' => 'Corolla',
            'year' => 2022,
            'color' => 'White',
            'license_plate' => 'LES-1234',
            'registration_number' => 'REG-' . strtoupper(uniqid()),
            'total_seats' => 4,
            'vehicle_images' => ['vehicles/corolla_white.png'],
            'is_verified' => true,
            'is_active' => true,
        ]);

        $vehicle2 = Vehicle::create([
            'driver_id' => $driver2->id,
            'vehicle_type' => 'sedan',
            'brand' => 'Honda',
            'model' => 'Civic',
            'year' => 2023,
            'color' => 'Silver',
            'license_plate' => 'LEA-5678',
            'registration_number' => 'REG-' . strtoupper(uniqid()),
            'total_seats' => 5,
            'vehicle_images' => ['vehicles/civic_silver.png'],
            'is_verified' => true,
            'is_active' => true,
        ]);

        $vehicle3 = Vehicle::create([
            'driver_id' => $driver1->id,
            'vehicle_type' => 'hatchback',
            'brand' => 'Suzuki',
            'model' => 'Cultus',
            'year' => 2021,
            'color' => 'Blue',
            'license_plate' => 'LHE-9012',
            'registration_number' => 'REG-' . strtoupper(uniqid()),
            'total_seats' => 4,
            'vehicle_images' => ['vehicles/cultus_blue.png'],
            'is_verified' => true,
            'is_active' => true,
        ]);

        $vehicle4 = Vehicle::create([
            'driver_id' => $admin->id,
            'vehicle_type' => 'sedan',
            'brand' => 'Toyota',
            'model' => 'Yaris',
            'year' => 2022,
            'color' => 'Red',
            'license_plate' => 'ISB-3456',
            'registration_number' => 'REG-' . strtoupper(uniqid()),
            'total_seats' => 4,
            'vehicle_images' => ['vehicles/yaris_red.png'],
            'is_verified' => true,
            'is_active' => true,
        ]);

        $vehicle5 = Vehicle::create([
            'driver_id' => $driver2->id,
            'vehicle_type' => 'suv',
            'brand' => 'Toyota',
            'model' => 'Fortuner',
            'year' => 2023,
            'color' => 'Black',
            'license_plate' => 'ISB-7890',
            'registration_number' => 'REG-' . strtoupper(uniqid()),
            'total_seats' => 7,
            'vehicle_images' => ['vehicles/fortuner_black.png'],
            'is_verified' => true,
            'is_active' => true,
        ]);

        // Create Routes - Popular Pakistan Routes
        $route1 = Route::create([
            'from_city_id' => $karachi->id,
            'to_city_id' => $lahore->id,
            'distance_km' => 1210.0,
            'estimated_duration_minutes' => 780,
            'base_price_per_seat' => 150.00,
            'is_popular' => true,
        ]);

        $route2 = Route::create([
            'from_city_id' => $lahore->id,
            'to_city_id' => $islamabad->id,
            'distance_km' => 375.0,
            'estimated_duration_minutes' => 240,
            'base_price_per_seat' => 60.00,
            'is_popular' => true,
        ]);

        $route3 = Route::create([
            'from_city_id' => $islamabad->id,
            'to_city_id' => $karachi->id,
            'distance_km' => 1405.0,
            'estimated_duration_minutes' => 900,
            'base_price_per_seat' => 180.00,
            'is_popular' => true,
        ]);

        $route4 = Route::create([
            'from_city_id' => $faisalabad->id,
            'to_city_id' => $multan->id,
            'distance_km' => 267.0,
            'estimated_duration_minutes' => 180,
            'base_price_per_seat' => 45.00,
            'is_popular' => true,
        ]);

        // Create Rides
        $ride1 = Ride::create([
            'driver_id' => $driver1->id,
            'vehicle_id' => $vehicle1->id,
            'route_id' => $route1->id,
            'departure_datetime' => now()->addDays(2)->setTime(9, 0),
            'arrival_datetime' => now()->addDays(2)->setTime(11, 0),
            'available_seats' => 3,
            'price_per_seat' => 50.00,
            'full_car_price' => 180.00,
            'ride_status' => 'scheduled',
            'pickup_point' => 'Times Square, Manhattan',
            'drop_point' => 'City Center, Philadelphia',
            'amenities' => ['AC', 'Music', 'WiFi'],
            'ride_preferences' => ['No Smoking', 'Quiet Ride'],
        ]);

        $ride2 = Ride::create([
            'driver_id' => $driver2->id,
            'vehicle_id' => $vehicle2->id,
            'route_id' => $route2->id,
            'departure_datetime' => now()->addDays(3)->setTime(14, 30),
            'arrival_datetime' => now()->addDays(3)->setTime(20, 30),
            'available_seats' => 4,
            'price_per_seat' => 90.00,
            'full_car_price' => 320.00,
            'ride_status' => 'scheduled',
            'pickup_point' => 'Downtown LA',
            'drop_point' => 'Phoenix Airport',
            'amenities' => ['AC', 'Music', 'Snacks'],
            'ride_preferences' => ['Pets Allowed', 'Music OK'],
        ]);

        $ride3 = Ride::create([
            'driver_id' => $driver1->id,
            'vehicle_id' => $vehicle3->id,
            'route_id' => $route3->id,
            'departure_datetime' => now()->addDays(5)->setTime(6, 0),
            'arrival_datetime' => now()->addDays(5)->setTime(19, 0),
            'available_seats' => 3,
            'price_per_seat' => 130.00,
            'full_car_price' => 450.00,
            'ride_status' => 'scheduled',
            'pickup_point' => 'F-7 Markaz, Islamabad',
            'drop_point' => 'Clifton, Karachi',
            'amenities' => ['AC', 'WiFi', 'Premium Seats'],
            'ride_preferences' => ['No Smoking', 'Professional'],
        ]);

        $ride4 = Ride::create([
            'driver_id' => $admin->id,
            'vehicle_id' => $vehicle4->id,
            'route_id' => $route4->id,
            'departure_datetime' => now()->addDays(1)->setTime(7, 0),
            'arrival_datetime' => now()->addDays(1)->setTime(10, 0),
            'available_seats' => 3,
            'price_per_seat' => 45.00,
            'full_car_price' => 150.00,
            'ride_status' => 'scheduled',
            'pickup_point' => 'Clock Tower, Faisalabad',
            'drop_point' => 'Ghanta Ghar, Multan',
            'amenities' => ['AC', 'Music'],
            'ride_preferences' => ['No Smoking'],
        ]);

        $ride5 = Ride::create([
            'driver_id' => $driver2->id,
            'vehicle_id' => $vehicle5->id,
            'route_id' => $route1->id,
            'departure_datetime' => now()->addDays(4)->setTime(5, 0),
            'arrival_datetime' => now()->addDays(4)->setTime(18, 0),
            'available_seats' => 6,
            'price_per_seat' => 170.00,
            'full_car_price' => 800.00,
            'ride_status' => 'scheduled',
            'pickup_point' => 'Saddar, Karachi',
            'drop_point' => 'Liberty Market, Lahore',
            'amenities' => ['AC', 'Music', 'WiFi', 'Snacks', 'Phone Charger'],
            'ride_preferences' => ['Family Friendly', 'Luggage Space'],
        ]);

        // Create Bookings
        $booking1 = Booking::create([
            'ride_id' => $ride1->id,
            'passenger_id' => $passenger1->id,
            'booking_type' => 'seat',
            'number_of_seats' => 1,
            'total_price' => 50.00,
            'payment_status' => 'paid',
            'booking_status' => 'confirmed',
            'pickup_location' => 'Times Square, Manhattan',
            'drop_location' => 'City Center, Philadelphia',
        ]);

        $booking2 = Booking::create([
            'ride_id' => $ride2->id,
            'passenger_id' => $passenger2->id,
            'booking_type' => 'seat',
            'number_of_seats' => 2,
            'total_price' => 180.00,
            'payment_status' => 'paid',
            'booking_status' => 'confirmed',
            'pickup_location' => 'Downtown LA',
            'drop_location' => 'Phoenix Airport',
            'passenger_notes' => 'Traveling with luggage',
        ]);

        // Create Payments
        Payment::create([
            'booking_id' => $booking1->id,
            'user_id' => $passenger1->id,
            'amount' => 50.00,
            'payment_method' => 'card',
            'transaction_id' => 'TXN-' . strtoupper(uniqid()),
            'payment_status' => 'completed',
            'payment_date' => now(),
        ]);

        Payment::create([
            'booking_id' => $booking2->id,
            'user_id' => $passenger2->id,
            'amount' => 180.00,
            'payment_method' => 'wallet',
            'transaction_id' => 'TXN-' . strtoupper(uniqid()),
            'payment_status' => 'completed',
            'payment_date' => now(),
        ]);

        // Create Reviews
        Review::create([
            'booking_id' => $booking1->id,
            'reviewer_id' => $passenger1->id,
            'reviewee_id' => $driver1->id,
            'rating' => 5,
            'comment' => 'Excellent driver! Very professional and on time.',
            'review_type' => 'driver_review',
        ]);

        Review::create([
            'booking_id' => $booking2->id,
            'reviewer_id' => $passenger2->id,
            'reviewee_id' => $driver2->id,
            'rating' => 5,
            'comment' => 'Great ride! Comfortable vehicle and smooth journey.',
            'review_type' => 'driver_review',
        ]);

        // Create Notifications
        Notification::create([
            'user_id' => $passenger1->id,
            'title' => 'Booking Confirmed',
            'message' => 'Your booking for New York → Philadelphia has been confirmed!',
            'type' => 'success',
            'is_read' => false,
        ]);

        Notification::create([
            'user_id' => $driver1->id,
            'title' => 'New Booking',
            'message' => 'You have a new booking from Mike Passenger for your ride.',
            'type' => 'info',
            'is_read' => false,
        ]);

        Notification::create([
            'user_id' => $passenger2->id,
            'title' => 'Payment Successful',
            'message' => 'Your payment of $180.00 has been processed successfully.',
            'type' => 'success',
            'is_read' => true,
        ]);

        $this->command->info('✅ Database seeded successfully!');
        $this->command->info('📧 Login Credentials:');
        $this->command->info('   Admin: admin@ridepool.com / password');
        $this->command->info('   Driver: john@driver.com / password');
        $this->command->info('   Passenger: mike@passenger.com / password');
    }
}
