@extends('layouts.app')

@section('title', 'Driver Dashboard')

@section('styles')
<style>
    /* Hero Header */
    .driver-hero {
        background: linear-gradient(135deg, #1e3a5f 0%, #0f172a 100%);
        padding: 40px 20px 60px;
        margin: -20px -20px 0;
        position: relative;
        overflow: hidden;
    }

    .driver-hero::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: url("data:image/svg+xml,%3Csvg width='60' height='60' viewBox='0 0 60 60' xmlns='http://www.w3.org/2000/svg'%3E%3Cg fill='none' fill-rule='evenodd'%3E%3Cg fill='%23ffffff' fill-opacity='0.03'%3E%3Cpath d='M36 34v-4h-2v4h-4v2h4v4h2v-4h4v-2h-4zm0-30V0h-2v4h-4v2h4v4h2V6h4V4h-4zM6 34v-4H4v4H0v2h4v4h2v-4h4v-2H6zM6 4V0H4v4H0v2h4v4h2V6h4V4H6z'/%3E%3C/g%3E%3C/g%3E%3C/svg%3E");
    }

    .hero-content {
        position: relative;
        z-index: 2;
        max-width: 1200px;
        margin: 0 auto;
    }

    .driver-profile-card {
        display: flex;
        align-items: center;
        gap: 20px;
        margin-bottom: 24px;
    }

    .driver-avatar {
        width: 80px;
        height: 80px;
        border-radius: 20px;
        background: linear-gradient(135deg, #00aff5, #10b981);
        display: flex;
        align-items: center;
        justify-content: center;
        color: white;
        font-weight: 800;
        font-size: 32px;
        border: 4px solid rgba(255,255,255,0.2);
    }

    .driver-info h1 {
        color: white;
        font-size: 28px;
        font-weight: 800;
        margin-bottom: 4px;
    }

    .driver-info p {
        color: rgba(255,255,255,0.6);
        font-size: 14px;
    }

    .driver-badge {
        display: inline-flex;
        align-items: center;
        gap: 6px;
        background: rgba(16, 185, 129, 0.2);
        color: #10b981;
        padding: 6px 12px;
        border-radius: 8px;
        font-size: 12px;
        font-weight: 700;
        margin-top: 8px;
    }

    .driver-badge.pending {
        background: rgba(245, 158, 11, 0.2);
        color: #f59e0b;
    }

    /* Quick Stats */
    .quick-stats {
        display: grid;
        grid-template-columns: repeat(4, 1fr);
        gap: 16px;
    }

    .quick-stat {
        background: rgba(255,255,255,0.1);
        backdrop-filter: blur(10px);
        border-radius: 16px;
        padding: 20px;
        text-align: center;
        border: 1px solid rgba(255,255,255,0.1);
    }

    .quick-stat-value {
        font-size: 28px;
        font-weight: 800;
        color: white;
    }

    .quick-stat-label {
        font-size: 12px;
        color: rgba(255,255,255,0.6);
        font-weight: 600;
        margin-top: 4px;
    }

    /* Main Content */
    .dashboard-content {
        max-width: 1200px;
        margin: -40px auto 0;
        padding: 0 20px 40px;
        position: relative;
        z-index: 3;
    }

    .dashboard-grid {
        display: grid;
        grid-template-columns: 2fr 1fr;
        gap: 24px;
    }

    .card {
        background: white;
        border-radius: 20px;
        box-shadow: 0 10px 40px rgba(0,0,0,0.08);
        overflow: hidden;
    }

    .card-header {
        padding: 20px 24px;
        border-bottom: 1px solid #f1f5f9;
        display: flex;
        justify-content: space-between;
        align-items: center;
    }

    .card-title {
        font-size: 18px;
        font-weight: 800;
        color: #1e3a5f;
    }

    .card-body {
        padding: 24px;
    }

    /* Bookings List */
    .booking-item {
        display: flex;
        align-items: center;
        gap: 16px;
        padding: 16px;
        background: #f8fafc;
        border-radius: 14px;
        margin-bottom: 12px;
        transition: all 0.2s;
    }

    .booking-item:hover {
        background: #f1f5f9;
        transform: translateX(4px);
    }

    .booking-avatar {
        width: 50px;
        height: 50px;
        border-radius: 12px;
        background: #e2e8f0;
        display: flex;
        align-items: center;
        justify-content: center;
        font-weight: 700;
        color: #64748b;
    }

    .booking-info {
        flex: 1;
    }

    .booking-route {
        font-weight: 700;
        color: #1e3a5f;
        font-size: 15px;
    }

    .booking-details {
        font-size: 13px;
        color: #64748b;
    }

    .booking-price {
        font-weight: 800;
        color: #10b981;
        font-size: 16px;
    }

    .booking-status {
        padding: 6px 12px;
        border-radius: 8px;
        font-size: 11px;
        font-weight: 700;
    }

    .status-confirmed {
        background: #d1fae5;
        color: #065f46;
    }

    .status-pending {
        background: #fef3c7;
        color: #92400e;
    }

    /* Earnings Card */
    .earnings-summary {
        text-align: center;
        padding: 20px;
    }

    .earnings-total {
        font-size: 42px;
        font-weight: 800;
        color: #10b981;
        line-height: 1;
    }

    .earnings-label {
        font-size: 13px;
        color: #64748b;
        margin-top: 8px;
    }

    .earnings-breakdown {
        margin-top: 20px;
        padding-top: 20px;
        border-top: 2px dashed #e2e8f0;
    }

    .earning-row {
        display: flex;
        justify-content: space-between;
        padding: 10px 0;
        font-size: 14px;
    }

    .earning-row span:first-child {
        color: #64748b;
    }

    .earning-row span:last-child {
        font-weight: 700;
        color: #1e3a5f;
    }

    /* Quick Actions */
    .quick-actions {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 12px;
    }

    .action-btn {
        display: flex;
        flex-direction: column;
        align-items: center;
        gap: 8px;
        padding: 20px;
        background: #f8fafc;
        border-radius: 14px;
        text-decoration: none;
        color: #1e3a5f;
        font-weight: 700;
        font-size: 13px;
        transition: all 0.2s;
        border: 2px solid transparent;
    }

    .action-btn:hover {
        background: #e0f7fa;
        border-color: #00aff5;
    }

    .action-btn .icon {
        width: 48px;
        height: 48px;
        background: linear-gradient(135deg, #00aff5, #10b981);
        border-radius: 12px;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 24px;
    }

    .empty-state {
        text-align: center;
        padding: 40px;
        color: #64748b;
    }

    .empty-state .icon {
        font-size: 48px;
        margin-bottom: 16px;
    }

    /* Mobile */
    @media (max-width: 768px) {
        .quick-stats {
            grid-template-columns: repeat(2, 1fr);
        }
        .dashboard-grid {
            grid-template-columns: 1fr;
        }
        .driver-hero {
            padding: 30px 16px 50px;
        }
    }
</style>
@endsection

@section('content')
<!-- Hero Header -->
<div class="driver-hero">
    <div class="hero-content">
        <div class="driver-profile-card">
            <div class="driver-avatar">
                @if($user->avatar)
                    <img src="{{ asset('storage/' . $user->avatar) }}" alt="" style="width: 100%; height: 100%; object-fit: cover; border-radius: 16px;">
                @else
                    {{ substr($user->name, 0, 1) }}
                @endif
            </div>
            <div class="driver-info">
                <h1>{{ $user->name }}</h1>
                <p>{{ $user->email }} • {{ $user->phone }}</p>
                @if($user->is_verified)
                    <div class="driver-badge">✅ Verified Driver</div>
                @else
                    <div class="driver-badge pending">⏳ Verification Pending</div>
                @endif
            </div>
        </div>

        <div class="quick-stats">
            <div class="quick-stat">
                <div class="quick-stat-value">{{ $stats['total_rides'] }}</div>
                <div class="quick-stat-label">Total Rides</div>
            </div>
            <div class="quick-stat">
                <div class="quick-stat-value">{{ $stats['completed_rides'] }}</div>
                <div class="quick-stat-label">Completed</div>
            </div>
            <div class="quick-stat">
                <div class="quick-stat-value">{{ $stats['pending_bookings'] }}</div>
                <div class="quick-stat-label">Pending Bookings</div>
            </div>
            <div class="quick-stat">
                <div class="quick-stat-value">⭐ {{ number_format($user->rating ?? 5.0, 1) }}</div>
                <div class="quick-stat-label">Rating</div>
            </div>
        </div>
    </div>
</div>

<!-- Main Content -->
<div class="dashboard-content">
    <div class="dashboard-grid">
        <!-- Left Column - Bookings -->
        <div>
            <div class="card" style="margin-bottom: 24px;">
                <div class="card-header">
                    <h3 class="card-title">📋 Recent Bookings</h3>
                    <a href="{{ route('dashboard') }}" style="color: #00aff5; font-weight: 700; font-size: 13px; text-decoration: none;">View All →</a>
                </div>
                <div class="card-body">
                    @forelse($recent_bookings as $booking)
                        <div class="booking-item">
                            <div class="booking-avatar">
                                {{ substr($booking->passenger->name ?? 'P', 0, 1) }}
                            </div>
                            <div class="booking-info">
                                <div class="booking-route">
                                    {{ $booking->ride->route->fromCity->name ?? 'N/A' }} → {{ $booking->ride->route->toCity->name ?? 'N/A' }}
                                </div>
                                <div class="booking-details">
                                    {{ $booking->passenger->name ?? 'Passenger' }} • {{ $booking->number_of_seats }} seat(s) • {{ $booking->created_at->format('M d') }}
                                </div>
                            </div>
                            <div class="booking-price">Rs {{ number_format($booking->total_price * 280) }}</div>
                            <span class="booking-status status-{{ $booking->booking_status == 'confirmed' ? 'confirmed' : 'pending' }}">
                                {{ ucfirst($booking->booking_status) }}
                            </span>
                        </div>
                    @empty
                        <div class="empty-state">
                            <div class="icon">📭</div>
                            <div style="font-weight: 700;">No bookings yet</div>
                            <div style="font-size: 13px;">Create a ride to start receiving bookings</div>
                        </div>
                    @endforelse
                </div>
            </div>

            <!-- Quick Actions -->
            <div class="card">
                <div class="card-header">
                    <h3 class="card-title">⚡ Quick Actions</h3>
                </div>
                <div class="card-body">
                    <div class="quick-actions">
                        <a href="{{ route('rides.create') }}" class="action-btn">
                            <div class="icon">🚗</div>
                            Offer a Ride
                        </a>
                        <a href="{{ route('profile') }}" class="action-btn">
                            <div class="icon">👤</div>
                            Edit Profile
                        </a>
                        <a href="{{ route('inbox') }}" class="action-btn">
                            <div class="icon">💬</div>
                            Messages
                        </a>
                        <a href="{{ route('dashboard') }}" class="action-btn">
                            <div class="icon">📊</div>
                            Full Dashboard
                        </a>
                    </div>
                </div>
            </div>
        </div>

        <!-- Right Column - Earnings & Stats -->
        <div>
            <div class="card" style="margin-bottom: 24px;">
                <div class="card-header">
                    <h3 class="card-title">💰 My Earnings</h3>
                </div>
                <div class="card-body">
                    <div class="earnings-summary">
                        <div class="earnings-total">Rs {{ number_format($earnings['total']) }}</div>
                        <div class="earnings-label">Total Earnings</div>

                        <div class="earnings-breakdown">
                            <div class="earning-row">
                                <span>This Month</span>
                                <span>Rs {{ number_format($earnings['this_month']) }}</span>
                            </div>
                            <div class="earning-row">
                                <span>Last Month</span>
                                <span>Rs {{ number_format($earnings['last_month']) }}</span>
                            </div>
                            <div class="earning-row">
                                <span>Pending Payout</span>
                                <span style="color: #f59e0b;">Rs {{ number_format($earnings['pending']) }}</span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- My Vehicles -->
            <div class="card">
                <div class="card-header">
                    <h3 class="card-title">🚙 My Vehicles</h3>
                    <a href="{{ route('profile') }}#vehicles" style="color: #00aff5; font-weight: 700; font-size: 13px; text-decoration: none;">Add +</a>
                </div>
                <div class="card-body">
                    @forelse($vehicles as $vehicle)
                        <div style="display: flex; align-items: center; gap: 12px; padding: 12px; background: #f8fafc; border-radius: 12px; margin-bottom: 10px;">
                            <div style="width: 50px; height: 50px; background: #e2e8f0; border-radius: 10px; display: flex; align-items: center; justify-content: center; font-size: 24px;">🚗</div>
                            <div style="flex: 1;">
                                <div style="font-weight: 700; color: #1e3a5f;">{{ $vehicle->make }} {{ $vehicle->model }}</div>
                                <div style="font-size: 12px; color: #64748b;">{{ $vehicle->license_plate }} • {{ $vehicle->color }}</div>
                            </div>
                            @if($vehicle->is_verified)
                                <span style="background: #d1fae5; color: #065f46; padding: 4px 10px; border-radius: 6px; font-size: 11px; font-weight: 700;">Verified</span>
                            @else
                                <span style="background: #fef3c7; color: #92400e; padding: 4px 10px; border-radius: 6px; font-size: 11px; font-weight: 700;">Pending</span>
                            @endif
                        </div>
                    @empty
                        <div class="empty-state" style="padding: 20px;">
                            <div>No vehicles added yet</div>
                            <a href="{{ route('profile') }}#vehicles" style="color: #00aff5; font-weight: 700;">Add Vehicle</a>
                        </div>
                    @endforelse
                </div>
            </div>
        </div>
    </div>
</div>
@endsection
